/**
 * Enhanced Notification System with Sound Support
 * This system provides beautiful notifications with sound effects
 * for success, error, warning, delete, and admin actions
 */
class NotificationSystem {
    constructor() {
        this.sounds = {
            success: '/assets/sounds/success.mp3',
            error: '/assets/sounds/error.mp3',
            warning: '/assets/sounds/warning.mp3',
            delete: '/assets/sounds/delete.mp3',
            admin: '/assets/sounds/admin.mp3',
            sale: '/assets/sounds/sale-complete.mp3'
        };
        
        this.isEnabled = true;
        this.soundEnabled = localStorage.getItem('notificationSounds') !== 'false';
        this.audioContextReady = false;
        this.audioContext = null;
        this.init();
        
        // Initialize audio context on first user interaction
        this.initAudioContext();
    }
    
    init() {
        // Create notification container if it doesn't exist
        if (!document.getElementById('notification-container')) {
            const container = document.createElement('div');
            container.id = 'notification-container';
            container.className = 'notification-container';
            document.body.appendChild(container);
        }
        
        // Add CSS styles
        this.addStyles();
        
        // Add sound toggle button - DISABLED
        // this.addSoundToggle();
    }
    
    addStyles() {
        if (document.getElementById('notification-styles')) return;
        
        const style = document.createElement('style');
        style.id = 'notification-styles';
        style.textContent = `
            .notification-container {
                position: fixed;
                top: 20px;
                right: 20px;
                z-index: 9999;
                max-width: 400px;
                pointer-events: none;
            }
            
            .notification {
                background: white;
                border-radius: 12px;
                padding: 16px 20px;
                margin-bottom: 12px;
                box-shadow: 0 8px 32px rgba(0, 0, 0, 0.12);
                border-left: 4px solid;
                display: flex;
                align-items: center;
                transform: translateX(100%);
                opacity: 0;
                transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
                pointer-events: auto;
                position: relative;
                overflow: hidden;
                backdrop-filter: blur(10px);
                border: 1px solid rgba(255, 255, 255, 0.2);
            }
            
            .notification.show {
                transform: translateX(0);
                opacity: 1;
            }
            
            .notification.hide {
                transform: translateX(100%);
                opacity: 0;
                margin-bottom: 0;
                padding-top: 0;
                padding-bottom: 0;
            }
            
            .notification::before {
                content: '';
                position: absolute;
                top: 0;
                left: 0;
                right: 0;
                height: 100%;
                background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.1), transparent);
                transform: translateX(-100%);
                animation: shine 2s ease-in-out;
            }
            
            @keyframes shine {
                0% { transform: translateX(-100%); }
                100% { transform: translateX(100%); }
            }
            
            .notification-icon {
                font-size: 24px;
                margin-right: 12px;
                min-width: 24px;
                display: flex;
                align-items: center;
                justify-content: center;
            }
            
            .notification-content {
                flex: 1;
                display: flex;
                flex-direction: column;
            }
            
            .notification-title {
                font-weight: 600;
                font-size: 16px;
                margin-bottom: 4px;
                line-height: 1.2;
            }
            
            .notification-message {
                font-size: 14px;
                opacity: 0.8;
                line-height: 1.3;
            }
            
            .notification-close {
                background: none;
                border: none;
                font-size: 18px;
                cursor: pointer;
                opacity: 0.5;
                padding: 0;
                margin-left: 8px;
                width: 20px;
                height: 20px;
                display: flex;
                align-items: center;
                justify-content: center;
                border-radius: 50%;
                transition: all 0.2s ease;
            }
            
            .notification-close:hover {
                opacity: 1;
                background: rgba(0, 0, 0, 0.1);
                transform: scale(1.1);
            }
            
            .notification-progress {
                position: absolute;
                bottom: 0;
                left: 0;
                height: 3px;
                background: rgba(255, 255, 255, 0.3);
                border-radius: 0 0 12px 12px;
                animation: progress linear;
            }
            
            @keyframes progress {
                0% { width: 100%; }
                100% { width: 0%; }
            }
            
            /* Success */
            .notification.success {
                border-left-color: #10b981;
                background: linear-gradient(135deg, #ecfdf5 0%, #f0fdf4 100%);
                color: #065f46;
            }
            .notification.success .notification-icon {
                color: #10b981;
            }
            
            /* Error */
            .notification.error {
                border-left-color: #ef4444;
                background: linear-gradient(135deg, #fef2f2 0%, #fef2f2 100%);
                color: #991b1b;
            }
            .notification.error .notification-icon {
                color: #ef4444;
            }
            
            /* Warning */
            .notification.warning {
                border-left-color: #f59e0b;
                background: linear-gradient(135deg, #fffbeb 0%, #fefce8 100%);
                color: #92400e;
            }
            .notification.warning .notification-icon {
                color: #f59e0b;
            }
            
            /* Info */
            .notification.info {
                border-left-color: #3b82f6;
                background: linear-gradient(135deg, #eff6ff 0%, #f0f9ff 100%);
                color: #1e40af;
            }
            .notification.info .notification-icon {
                color: #3b82f6;
            }
            
            /* Delete */
            .notification.delete {
                border-left-color: #dc2626;
                background: linear-gradient(135deg, #fef2f2 0%, #fee2e2 100%);
                color: #7f1d1d;
            }
            .notification.delete .notification-icon {
                color: #dc2626;
            }
            
            /* Admin */
            .notification.admin {
                border-left-color: #8b5cf6;
                background: linear-gradient(135deg, #f5f3ff 0%, #faf5ff 100%);
                color: #5b21b6;
            }
            .notification.admin .notification-icon {
                color: #8b5cf6;
            }
            
            /* Sale Complete */
            .notification.sale {
                border-left-color: #059669;
                background: linear-gradient(135deg, #d1fae5 0%, #a7f3d0 100%);
                color: #047857;
                font-size: 16px;
                padding: 20px 24px;
            }
            .notification.sale .notification-icon {
                color: #059669;
                font-size: 28px;
            }
            .notification.sale .notification-title {
                font-size: 18px;
                font-weight: 700;
            }
            
            /* Category/Brand Added */
            .notification.category {
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                border-left: 5px solid #667eea;
            }
            
            .notification.category .notification-icon {
                color: #667eea;
                background: rgba(102, 126, 234, 0.1);
            }
            
            .notification.category .notification-title {
                color: #4c63d2;
                font-weight: bold;
            }
            
            /* Sound toggle button - DISABLED */
            /*
            .sound-toggle {
                position: fixed;
                bottom: 20px;
                right: 20px;
                width: 50px;
                height: 50px;
                border-radius: 50%;
                background: #4f46e5;
                color: white;
                border: none;
                cursor: pointer;
                box-shadow: 0 4px 12px rgba(79, 70, 229, 0.3);
                transition: all 0.3s ease;
                z-index: 9998;
                display: flex;
                align-items: center;
                justify-content: center;
                font-size: 18px;
            }
            
            .sound-toggle:hover {
                background: #3730a3;
                transform: translateY(-2px);
                box-shadow: 0 6px 16px rgba(79, 70, 229, 0.4);
            }
            
            .sound-toggle.muted {
                background: #6b7280;
            }
            
            .sound-toggle.muted:hover {
                background: #4b5563;
            }
            */
            
            /* Mobile responsiveness */
            @media (max-width: 480px) {
                .notification-container {
                    left: 10px;
                    right: 10px;
                    max-width: none;
                }
                
                .notification {
                    transform: translateY(-100%);
                }
                
                .notification.show {
                    transform: translateY(0);
                }
                
                .notification.hide {
                    transform: translateY(-100%);
                }
                
                /*
                .sound-toggle {
                    width: 45px;
                    height: 45px;
                    bottom: 15px;
                    right: 15px;
                }
                */
            }
            
            /* Pulse animation for important notifications */
            .notification.pulse {
                animation: pulse 2s infinite;
            }
            
            @keyframes pulse {
                0% { transform: scale(1); }
                50% { transform: scale(1.02); }
                100% { transform: scale(1); }
            }
        `;
        document.head.appendChild(style);
    }
    
    addSoundToggle() {
        // DISABLED - Sound toggle icon removed per user request
        // Sound functionality remains active, only icon is hidden
        return;
        
        /*
        if (document.getElementById('sound-toggle-btn')) return;
        
        const button = document.createElement('button');
        button.id = 'sound-toggle-btn';
        button.className = `sound-toggle ${this.soundEnabled ? '' : 'muted'}`;
        button.innerHTML = this.soundEnabled ? '<i class="fas fa-volume-up"></i>' : '<i class="fas fa-volume-mute"></i>';
        button.title = this.soundEnabled ? 'Click to mute notifications' : 'Click to enable notification sounds';
        
        button.addEventListener('click', () => {
            this.soundEnabled = !this.soundEnabled;
            localStorage.setItem('notificationSounds', this.soundEnabled.toString());
            button.className = `sound-toggle ${this.soundEnabled ? '' : 'muted'}`;
            button.innerHTML = this.soundEnabled ? '<i class="fas fa-volume-up"></i>' : '<i class="fas fa-volume-mute"></i>';
            button.title = this.soundEnabled ? 'Click to mute notifications' : 'Click to enable notification sounds';
            
            // Show feedback
            this.show({
                type: 'info',
                title: 'Sound Settings',
                message: this.soundEnabled ? 'Notification sounds enabled' : 'Notification sounds muted',
                duration: 2000
            });
        });
        
        document.body.appendChild(button);
        */
    }
    
    initAudioContext() {
        // Initialize audio context on first user interaction
        const initAudio = () => {
            if (!this.audioContextReady) {
                try {
                    this.audioContext = new (window.AudioContext || window.webkitAudioContext)();
                    this.audioContextReady = true;
                    console.log('Audio context initialized successfully');
                    document.removeEventListener('click', initAudio);
                    document.removeEventListener('keydown', initAudio);
                    document.removeEventListener('touchstart', initAudio);
                } catch (error) {
                    console.warn('Failed to initialize audio context:', error);
                }
            }
        };
        
        document.addEventListener('click', initAudio);
        document.addEventListener('keydown', initAudio);
        document.addEventListener('touchstart', initAudio);
    }
    
    playSound(type) {
        console.log('playSound called with type:', type, 'soundEnabled:', this.soundEnabled, 'audioContextReady:', this.audioContextReady);
        if (!this.soundEnabled) {
            console.log('Sound is disabled, not playing');
            return;
        }
        
        if (!this.audioContextReady || !this.audioContext) {
            console.log('Audio context not ready, trying to initialize...');
            try {
                this.audioContext = new (window.AudioContext || window.webkitAudioContext)();
                this.audioContextReady = true;
                console.log('Audio context created directly');
            } catch (error) {
                console.warn('Cannot create audio context without user interaction:', error);
                return;
            }
        }
        
        try {
            // Use the initialized audio context
            const audioContext = this.audioContext;
            const oscillator = audioContext.createOscillator();
            const gainNode = audioContext.createGain();
            
            oscillator.connect(gainNode);
            gainNode.connect(audioContext.destination);
            
            // Different tones for different notification types
            const frequencies = {
                success: [523.25, 659.25], // C5, E5
                error: [415.30, 311.13],   // G#4, E♭4
                warning: [440, 554.37],    // A4, C#5
                delete: [369.99, 293.66], // F#4, D4
                admin: [587.33, 698.46],   // D5, F5
                sale: [523.25, 659.25, 783.99], // C5, E5, G5
                category: [440, 523.25, 659.25] // A4, C5, E5 - pleasant chord for category additions
            };
            
            const freq = frequencies[type] || frequencies.info;
            
            // Play chord or sequence
            if (type === 'sale') {
                // Play a triumphant chord progression
                freq.forEach((f, i) => {
                    setTimeout(() => {
                        const osc = audioContext.createOscillator();
                        const gain = audioContext.createGain();
                        
                        osc.connect(gain);
                        gain.connect(audioContext.destination);
                        
                        osc.frequency.setValueAtTime(f, audioContext.currentTime);
                        osc.type = 'sine';
                        
                        gain.gain.setValueAtTime(0, audioContext.currentTime);
                        gain.gain.linearRampToValueAtTime(0.1, audioContext.currentTime + 0.1);
                        gain.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.5);
                        
                        osc.start(audioContext.currentTime);
                        osc.stop(audioContext.currentTime + 0.5);
                    }, i * 100);
                });
            } else if (type === 'category') {
                // Play a pleasant chord for category additions
                freq.forEach((f, i) => {
                    setTimeout(() => {
                        const osc = audioContext.createOscillator();
                        const gain = audioContext.createGain();
                        
                        osc.connect(gain);
                        gain.connect(audioContext.destination);
                        
                        osc.frequency.setValueAtTime(f, audioContext.currentTime);
                        osc.type = 'sine';
                        
                        gain.gain.setValueAtTime(0, audioContext.currentTime);
                        gain.gain.linearRampToValueAtTime(0.08, audioContext.currentTime + 0.05);
                        gain.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.4);
                        
                        osc.start(audioContext.currentTime);
                        osc.stop(audioContext.currentTime + 0.4);
                    }, i * 60);
                });
            } else {
                // Simple tone
                oscillator.frequency.setValueAtTime(freq[0], audioContext.currentTime);
                oscillator.type = type === 'error' ? 'sawtooth' : 'sine';
                
                gainNode.gain.setValueAtTime(0, audioContext.currentTime);
                gainNode.gain.linearRampToValueAtTime(0.1, audioContext.currentTime + 0.1);
                gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.3);
                
                oscillator.start(audioContext.currentTime);
                oscillator.stop(audioContext.currentTime + 0.3);
            }
        } catch (error) {
            console.warn('Could not play notification sound:', error);
        }
    }
    
    show(options) {
        const {
            type = 'info',
            title = '',
            message = '',
            duration = 5000,
            persistent = false,
            pulse = false
        } = options;
        
        // Play sound
        this.playSound(type);
        
        // Create notification element
        const notification = document.createElement('div');
        notification.className = `notification ${type}${pulse ? ' pulse' : ''}`;
        
        // Get icon for notification type
        const icons = {
            success: '<i class="fas fa-check-circle"></i>',
            error: '<i class="fas fa-times-circle"></i>',
            warning: '<i class="fas fa-exclamation-triangle"></i>',
            info: '<i class="fas fa-info-circle"></i>',
            delete: '<i class="fas fa-trash"></i>',
            admin: '<i class="fas fa-crown"></i>',
            sale: '<i class="fas fa-cash-register"></i>',
            category: '<i class="fas fa-tags"></i>'
        };
        
        notification.innerHTML = `
            <div class="notification-icon">${icons[type] || icons.info}</div>
            <div class="notification-content">
                ${title ? `<div class="notification-title">${title}</div>` : ''}
                ${message ? `<div class="notification-message">${message}</div>` : ''}
            </div>
            <button class="notification-close" onclick="notificationSystem.hide(this.parentElement)">
                <i class="fas fa-times"></i>
            </button>
            ${!persistent ? `<div class="notification-progress" style="animation-duration: ${duration}ms;"></div>` : ''}
        `;
        
        // Add to container
        const container = document.getElementById('notification-container');
        container.appendChild(notification);
        
        // Show with animation
        setTimeout(() => {
            notification.classList.add('show');
        }, 10);
        
        // Auto hide if not persistent
        if (!persistent) {
            setTimeout(() => {
                this.hide(notification);
            }, duration);
        }
        
        return notification;
    }
    
    hide(notification) {
        if (!notification || !notification.parentElement) return;
        
        notification.classList.add('hide');
        setTimeout(() => {
            if (notification.parentElement) {
                notification.parentElement.removeChild(notification);
            }
        }, 400);
    }
    
    // Convenience methods
    success(title, message, options = {}) {
        return this.show({
            type: 'success',
            title,
            message,
            ...options
        });
    }
    
    error(title, message, options = {}) {
        return this.show({
            type: 'error',
            title,
            message,
            duration: 8000,
            ...options
        });
    }
    
    warning(title, message, options = {}) {
        return this.show({
            type: 'warning',
            title,
            message,
            ...options
        });
    }
    
    info(title, message, options = {}) {
        return this.show({
            type: 'info',
            title,
            message,
            ...options
        });
    }
    
    delete(title, message, options = {}) {
        return this.show({
            type: 'delete',
            title,
            message,
            duration: 4000,
            ...options
        });
    }
    
    admin(title, message, options = {}) {
        return this.show({
            type: 'admin',
            title,
            message,
            pulse: true,
            ...options
        });
    }
    
    // Modern confirmation dialog
    confirm(options) {
        const {
            title = 'Confirm Action',
            message = 'Are you sure?',
            confirmText = 'Confirm',
            cancelText = 'Cancel',
            type = 'warning',
            onConfirm = () => {},
            onCancel = () => {}
        } = options;

        // Create modal backdrop
        const backdrop = document.createElement('div');
        backdrop.className = 'confirmation-backdrop';
        backdrop.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            z-index: 10000;
            display: flex;
            align-items: center;
            justify-content: center;
            animation: fadeIn 0.2s ease;
        `;

        // Create confirmation dialog
        const dialog = document.createElement('div');
        dialog.className = 'confirmation-dialog';
        dialog.style.cssText = `
            background: white;
            border-radius: 8px;
            padding: 24px;
            max-width: 400px;
            width: 90%;
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.3);
            animation: slideIn 0.3s ease;
        `;

        const icons = {
            warning: '<i class="fas fa-exclamation-triangle" style="color: #f59e0b;"></i>',
            danger: '<i class="fas fa-trash" style="color: #ef4444;"></i>',
            info: '<i class="fas fa-info-circle" style="color: #3b82f6;"></i>'
        };

        dialog.innerHTML = `
            <div style="text-align: center; margin-bottom: 20px;">
                <div style="font-size: 48px; margin-bottom: 16px;">
                    ${icons[type] || icons.warning}
                </div>
                <h3 style="margin: 0 0 8px 0; color: #374151;">${title}</h3>
                <p style="margin: 0; color: #6b7280; line-height: 1.5;">${message}</p>
            </div>
            <div style="display: flex; gap: 12px; justify-content: center;">
                <button class="confirm-btn" style="
                    background: #ef4444;
                    color: white;
                    border: none;
                    padding: 10px 20px;
                    border-radius: 6px;
                    cursor: pointer;
                    font-weight: 600;
                    transition: all 0.2s;
                ">${confirmText}</button>
                <button class="cancel-btn" style="
                    background: #f3f4f6;
                    color: #374151;
                    border: none;
                    padding: 10px 20px;
                    border-radius: 6px;
                    cursor: pointer;
                    font-weight: 600;
                    transition: all 0.2s;
                ">${cancelText}</button>
            </div>
        `;

        // Add animations
        const style = document.createElement('style');
        style.textContent = `
            @keyframes fadeIn {
                from { opacity: 0; }
                to { opacity: 1; }
            }
            @keyframes slideIn {
                from { transform: scale(0.9) translateY(-20px); opacity: 0; }
                to { transform: scale(1) translateY(0); opacity: 1; }
            }
            .confirm-btn:hover {
                background: #dc2626 !important;
            }
            .cancel-btn:hover {
                background: #e5e7eb !important;
            }
        `;
        document.head.appendChild(style);

        backdrop.appendChild(dialog);
        document.body.appendChild(backdrop);

        // Handle button clicks
        dialog.querySelector('.confirm-btn').addEventListener('click', () => {
            document.body.removeChild(backdrop);
            document.head.removeChild(style);
            onConfirm();
        });

        dialog.querySelector('.cancel-btn').addEventListener('click', () => {
            document.body.removeChild(backdrop);
            document.head.removeChild(style);
            onCancel();
        });

        // Handle backdrop click
        backdrop.addEventListener('click', (e) => {
            if (e.target === backdrop) {
                document.body.removeChild(backdrop);
                document.head.removeChild(style);
                onCancel();
            }
        });

        // Handle escape key
        const handleEscape = (e) => {
            if (e.key === 'Escape') {
                document.body.removeChild(backdrop);
                document.head.removeChild(style);
                document.removeEventListener('keydown', handleEscape);
                onCancel();
            }
        };
        document.addEventListener('keydown', handleEscape);
    }

    // Convenience methods for common notifications
    showSuccess(message, title = 'Success') {
        return this.success(title, message);
    }
    
    showError(message, title = 'Error') {
        return this.error(title, message);
    }
    
    showWarning(message, title = 'Warning') {
        return this.warning(title, message);
    }
    
    showInfo(message, title = 'Info') {
        return this.info(title, message);
    }
    
    categoryAdded(message, title = 'Category Added') {
        return this.show({
            type: 'category',
            title,
            message,
            duration: 4000,
            pulse: true
        });
    }
    
    saleComplete(title, message, options = {}) {
        console.log('saleComplete called with:', { title, message, options });
        return this.show({
            type: 'sale',
            title,
            message,
            duration: 6000,
            pulse: true,
            ...options
        });
    }
    
    // Clear all notifications
    clear() {
        const container = document.getElementById('notification-container');
        if (container) {
            Array.from(container.children).forEach(notification => {
                this.hide(notification);
            });
        }
    }
}

// Initialize global notification system
window.notificationSystem = new NotificationSystem();

// Export for use in modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = NotificationSystem;
}